/***********************************************************************
   Xpress Optimizer Examples
   =========================

   file MipSolEnum.cs
   ``````````````````
   Find the 10 best solutions with the MIP solution enumerator

   We take the power generation problem stored in hpw15.mps which seeks to
   optimise the operating pattern of a group of electricity generators. We
   run the MIP solution enumerator on the problem using the default setup
   obtaining the best 10 solutions. The best 10 solutions are stored to a
    MIP solution pool.  The solutions' objectives and solution values are
   printed to screen.

   (c) 2008-2025 Fair Isaac Corporation
***********************************************************************/

using System;
using System.IO;
using Optimizer;

namespace XPRSExamples
{
    class MipSolEnum
    {
        public static void Main(string[] args)
        {
            XPRS.Init();

            XPRSprob prob = new XPRSprob();
            prob.AddMsgHandlerCallback(Console.Out);

            XPRSmipsolpool msp = new XPRSmipsolpool();
            XPRSmipsolenum mse = new XPRSmipsolenum();

            prob.MPSFormat = -1;
            prob.ReadProb("hpw15");
            int nCols = prob.Cols;

            /* Avoid duplicated solutions from heuristics. */
            msp.DuplicateSolutionsPolicy = 3;

            /* Disable dual reductions to prevent dominated solutions from */
            /* being presolved away. */
            prob.MIPDualReductions = 2;

            /* Run the enumeration */
            int nMaxSols = 10;
            mse.Minim(prob, msp, XPRSdefaultMipSolEnumHandler.GetDefaultHandlerCallback(), null, ref nMaxSols);

            /* Print out the solutions found */
            int nSols = mse.Solutions;
            for (int i = 1; i <= nSols; i++)
            {
                int[] iSolutionId = new int[1];
                double dObj;
                int nSols2, nReturnedSolIds, solutionIdStatus;
                mse.GetSolList((int)XPRSattribute.Mse_Metric_MipObject, i, i, iSolutionId, out nReturnedSolIds, out nSols2);
                mse.GetSolMetric(iSolutionId[0], out solutionIdStatus, (int)XPRSattribute.Mse_Metric_MipObject, out dObj);
                Console.WriteLine("--------\n" + i + " = " + dObj);
                for (int j = 0; j < nCols; j++)
                {
                    double[] dSol = new double[1];
                    int nValuesReturned;
                    msp.GetSol(iSolutionId[0], out solutionIdStatus, dSol, j, j, out nValuesReturned);
                    Console.WriteLine(j + " = " + dSol[0]);
                }
            }

            XPRS.Free();
        }
    }
}
